<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Common;

use Erlage\Photogram\System;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowEnum;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagFollowTable;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowModel;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowModel;

final class CommonQueries
{
    public static function beansWithMatchingPredicates(
        string $tableName,
        array $predicates,
        string $limit = null
    ): array {
        $abstractTableClass = System::getTableClassFromTableName($tableName);

        $query = (new Query()) -> from($abstractTableClass::getTableName());

        foreach ($predicates as $predicateFieldName => $predicateValuesToMatch)
        {
            if (\is_array($predicateValuesToMatch))
            {
                $query -> in($predicateFieldName, $predicateValuesToMatch);
            }
            else
            {
                $query -> where($predicateFieldName, $predicateValuesToMatch);
            }
        }

        if (null !== $limit)
        {
            $query -> limit($limit);
        }

        return $query -> select();
    }

    public static function modelsWithMatchingPredicates(string $tableName, array $predicates): array
    {
        $models = array();

        $abstractModelClass = System::getModelClassFromTableName($tableName);

        foreach (self::beansWithMatchingPredicates($tableName, $predicates) as $bean)
        {
            $model = $abstractModelClass::createFromUntouchedBean_noException($bean);

            if ($model -> isModel())
            {
                $models[] = $model;
            }
        }

        return $models;
    }

    /**
     * @return string[]
    */
    public static function userFollowingIds(string $userId): array
    {
        $followUserIds = array();

        /**
         * @var UserFollowModel[]
         */
        $modelsFromQuery = self::modelsWithMatchingPredicates(
            UserFollowTable::getTableName(),
            array(
                UserFollowTable::FOLLOWED_BY_USER_ID => $userId,
            )
        );

        foreach ($modelsFromQuery  as $model)
        {
            $followUserIds[] = $model -> getFollowedUserId();
        }

        return $followUserIds;
    }

    /**
     * @return string[]
    */
    public static function userHashtagFollowingIds(string $userId): array
    {
        $followHashtagIds = array();

        /**
         * @var HashtagFollowModel[]
         */
        $modelsFromQuery = self::modelsWithMatchingPredicates(
            HashtagFollowTable::getTableName(),
            array(
                HashtagFollowTable::FOLLOWED_BY_USER_ID => $userId,
            )
        );

        foreach ($modelsFromQuery  as $model)
        {
            $followHashtagIds[] = $model -> getFollowedHashtagId();
        }

        return $followHashtagIds;
    }

    public static function isFollowing(string $followedUserId, string $followedByUserId): bool
    {
        $followBean = (new Query())
            -> from(UserFollowTable::TABLE_NAME)
            -> where(UserFollowTable::FOLLOWED_USER_ID, $followedUserId)
            -> where(UserFollowTable::FOLLOWED_BY_USER_ID, $followedByUserId)
            -> where(UserFollowTable::META_IS_PENDING, UserFollowEnum::META_IS_PENDING_NO)
            -> selectOne();

        return null != $followBean ? true : false;
    }
}
